//=============================================================================
// MYTH_CGC_ActionPack1
//=============================================================================

/*:
 * @target MZ *
 * @author Isiah Brighton
 * @plugindesc v1.2.1 An extension to MYTH_CGC_CoreEngine that adds several new Card Actions.
 * @url https://www.patreon.com/mythatelier
 *
 * @param selectionHighlight
 * @text Selection Highlight Image
 * @default selectHighlight
 * @desc Image that appears on the Card when queued in the Selection Window. Must be in img/CGC/highlights.
 *
 * @param showUpperText
 * @text Show Upper Text?
 * @type boolean
 * @default true
 * @desc Toggles if the Upper Text appears in the Card Selection Window.
 * 
 * @param selectUpperTextString
 * @text Upper Text String
 * @type string
 * @default Choose %selectReq cards from %actor's %zone
 * @desc Default text for Upper Text string.
 * 
 * @param showLowerText
 * @text Show Lower Text?
 * @type boolean
 * @default true
 * @desc Toggles if the Lower Text appears in the Card Selection Window.
 * 
 * @param selectLowerTextString
 * @text Lower Text String
 * @type string
 * @default %selectNum / %selectReq selected
 * @desc Default text for Lower Text string.
 * 
 * @param zoneNames
 * @type struct<zoneNames>
 * @text Zone Display Names
 * @desc Text to display the names of the 3 default Zones. All Extra Zones will use the names you give them.
 * @default {"deck":"Deck","discard":"Discard","hand":"Hand"}
 * 
 * @help
 *
 * ============================================================================
 * Overview
 * ============================================================================
 *
 * This plugin adds several new Card Actions which involve the Card Selection
 * Window.
 * 
 * These Card Actions (CAs) work together in a group as part of 1 of 3 
 * stages:
 * 1. Selection
 * 2. Movement
 * 3. Clear
 * 
 * 
 * Selection CAs open the Selection Window and have the player select cards
 *     (e.g, "Select 2 from deck")
 * 
 * Movement CAs perform an action involving the cards selected in the 
 *     Selection CA (e.g, "Move selected to discard")
 *     
 * The Clear CA clears the list of selected cards so future Movement CAs do
 *     not use them ("Clear selected")
 *     
 * A Selection Card Action must be called before a Movement Card Action can
 * execute, and a Clear Card Action cleans up all internal variables to
 * prepare for the next set of Card Actions.
 * 
 * Example:
 *     <Card Actions>
 *     Select 2 from deck
 *     Move selected to discard
 *     Clear selected
 *     </Card Actions>
 * This example makes the player move 2 cards of their choice from their deck
 * to their discard.
 * 
 * 
 *
 * ============================================================================
 * Selection Card Actions
 * ============================================================================
 *
 * Select X from zoneName
 *     Replace 'X' with a number or game variable, and replace 'zoneName' with
 *     'deck', 'discard' or 'hand'
 *     This opens the Card Selection Window and makes the player choose X cards
 *     from the specified zone.
 *     
 * Select X of Y from zoneName location
 *     Replace 'X' with a number, 'Y' with another number,
 *     'zoneName' with a zone, and replace 'location' with either 'Top', 
 *     'Bottom' or 'Random'.
 *     This makes the player choose X cards from a selection of Y cards from
 *     the specified zone. If 'Top', The pool of cards will be grabbed from
 *     the top Y cards. If 'Bottom', the bottom. If 'random', cards will be
 *     grabbed randomly from the zone.
 *     
 * Select X from skills Y, Z, A, B
 *     Replace X, Y, Z, etc with numbers.
 *     This makes the player choose X cards from a list of Skill IDs.
 *     The list of IDs can be any size, separated by commas.
 * 
 * Select Any from zoneName
 *     Allows players to choose any number of cards from a specified zone 
 *     in a selection and then when they cancel/back out allows you to take 
 *     that selection and perform
 *     Card Actions on them
 * 
 *  Select All from zoneName
 *     This automatically loads all cards in the specified zone into Selected.
 * 
 * Select Upper Text: string
 *     Changes the Upper Text of the Selection Window to the string provided.
 * 
 * Select Lower Text: string
 *     Changes the Lower Text of the Selection Window to the string provided.
 * 
 * %actor - Actor Name
 * %selectNum - Number of cards currently selected
 * %unselectNum - Number of cards currently not selected
 * %selectReq - Number of cards required to be selected
 * %zone - Zone cards are being selected from (if any)
 * 
 * If you're using CardTypes:
 * 
 * 
 * Select X Type typeName from zoneName
 *     Replace X with a number, typeName with the name of a Card Type, and
 *     zoneName with a zone.
 *     This makes the player choose X cards from the specified zone filtered
 *     to only include the specified Card Type.
 *     
 * Select X of Y Type typeName from zoneName location
 *     This makes the player choose X cards from the specified zone,
 *     filtered to only include the specified Card Type, with a pool size
 *     of Y grabbed from the top/bottom or grabbed randomly from the zone.
 *   
 * 
 * Note that each Selection CA adds to the list of selected cards, rather than
 * overwriting it.
 *
 * ============================================================================
 * Movement Card Actions
 * ============================================================================
 *
 * Once the player performed a Selection, these Movement Card Actions will
 * take those selected Cards and do something with them.
 * 
 * Move selected to zoneName
 *     Replace 'zoneName' with a zone.
 *     This will move the selected cards from their previous zone to the
 *     specified zone.
 *     
 * Remove selected
 *     This will remove the selected cards from play.
 *     
 * Add selected to zoneName
 *     Replace 'zoneName' with a zone.
 *     This will add a copy of each selected card to the specified zone.
 *     Note that this is the only Movement Card Action that works when
 *     using the Card Action "Select from Skills" because it's the only one
 *     that does not require a zone of origin.
 *     
 * Move unselected to zoneName
 *     Replace 'zoneName' with a zone.
 *     This will move the cards that WEREN'T selected cards from their previous 
 *     zone to the specified zone.
 * 
 * Remove unselected
 *     This will remove the cards that WEREN'T selected from play.
 * 
 * Add unselected to zoneName
 *     Replace 'zoneName' with a zone.
 *     This will add a copy of each card that WASN'T selected to the specified 
 *     zone.
 *     Note that this is the only Movement Card Action that works when
 *     using the Card Action "Select from Skills" because it's the only one
 *     that does not require a zone of origin.
 *
 * ============================================================================
 * Clear Card Action
 * ============================================================================
 * Each Selection Card Action adds to the list of selected cards, and that
 * list will not be cleared until you call one of the following:
 * 
 * Clear Selected
 * Clear Selection
 * 
 * This means that, until you call Clear Selected/Selection, you can perform
 * multiple Selection CAs and one Movement CA which will act on all of them,
 * or call multiple movement CAs with only one selected card.
 * 
* ============================================================================
* For more information and other features please use the wiki
* that has been made for this and all related plugins:
* http://card-game-combat-help.alwaysdata.net/
* ============================================================================
 * 
 * 
 * ============================================================================
 * Version History
 * ============================================================================
 * 
 * v1.2.1 - Added Message Warning infrastructure.
 * 
 * v1.2.0 - Added Select Any from zoneName Action
 *          Added Select All Type typeName from zoneName
 *          Added "unselected" variant of Movement Card Actions
 *          Added Select Upper Text and Select Lower Text Card Actions.
 *          Moved highlight sprite loading to img/CGC/highlights
 *          Exposed text components to plugin parameters.
 *          Fixed crash when Removing Selected cards by type and you had fewer
 *          cards than the Card Action allowed for.
 *  
 * v1.1.1 - Fixed crash if selecting from a pool of cards that has a length of
 *          0.
 * 
 * v1.1.0 - Added Select All Card Action
 *          Selection text will no longer appear while browsing cards normally.
 *          Updated for combatibility with CGC core v1.6.0 and Card Summons.
 *          Minor refactoring to improve Card Actions reading variables
 *          Renamed plugin
 *          Added option to not display text in the Card Selection Window
 *          Fixed crash from selecting cards too fast.
 *          
 *
 * v1.0.0 - Released plugin
 *
 * ============================================================================
 * Contact Info
 * ============================================================================
 *
 * This tool was developed by folks at MythAtelier LLC. We make Games that Care.
 *
* Need more tools for your project? Be sure to check out our other plugins here:
* https://itch.io/c/1695699/tools-plugins
*
* Have any questions? Run into any bugs? Want to chat? Best place to reach us:
* https://discord.gg/wRk4XHF5tZ
*
* If you like this plugin and want to support us, please give our Patreon a look:
* https://www.patreon.com/mythatelier
*
* Other Places to Find Us:
* - YouTube: https://www.youtube.com/channel/UCQn_1kWmj2iAkNEv9eqL_LA
* - BSky: https://bsky.app/profile/mythatelier.itch.io
* - Steam: https://store.steampowered.com/search/?developer=MythAtelier%2C%20LLC
*
 */

/*~struct~zoneNames:
* 
* @param deck
* @text Deck
* @desc Display name for the Deck Zone in the Selection Window.
* @default Deck
* 
* @param discard
* @text Discard
* @desc Display name for the Discard Zone in the Selection Window.
* @default Discard
* 
* @param hand
* @text Hand
* @desc Display name for the Hand Zone in the Selection Window.
* @default Hand
* 
*/

var Myth = Myth || {};
if (!Myth.CGC)
	console.error("MYTH_CGC_ActionPack1 should be placed underneath MYTH_CGC_CoreEngine.");
Myth.CGC.AP = Myth.CGC.AP || {};
Myth.Parameters = PluginManager.parameters('MYTH_CGC_ActionPack1');

//=============================================================================
// Warn Messages
//=============================================================================
// #region Warn Messages

Myth.CGC.AP.warnMessage = function (index, ...args)
{
	if ($dataSystem == undefined)
	{
		return setTimeout(() =>
		{
			this.warnMessage(index, ...args);
		}, 100);
	}
	const languagePrefix = $dataSystem.locale.slice(0, 2).toLowerCase();
	let text = "";
	let warnMessageMap = {
		ja: this.warnMessageJa,		// Japanese
		zh: this.warnMessageZh,		// Chinese
		ko: this.warnMessageKo,		// Korean
		en: this.warnMessageEn		// English
	};
	if (warnMessageMap[languagePrefix])
		text = warnMessageMap[languagePrefix].call(this, index, ...args);
	else
	{
		text = this.warnMessageEn(index, ...args);	// Default to English for any unsupported language.
	}

	console.warn(text);
	console.warn("MythAtelier Error Code: AP " + index);  // Error code should be the same in every language.
	Myth.Util.openConsole();
	return text;
}

Myth.CGC.AP.warnMessageEn = function (index, ...args)
{
	let text = "";
	return text;
}

Myth.CGC.AP.warnMessageJa = function (index, ...args)
{
	let text = "";
	return text;
}

Myth.CGC.AP.warnMessageKo = function (index, ...args)
{
	let text = "";
	return text;
}

Myth.CGC.AP.warnMessageZh = function (index, ...args)
{
	let text = "";
	return text;
}

// #endregion

Myth.CGC.images.cardSelectedHighlight = Myth.Parameters.selectionHighlight;
Myth.CGC.showSelectionUpperText = JSON.parse(Myth.Parameters.showUpperText);
Myth.CGC.showSelectionLowerText = JSON.parse(Myth.Parameters.showLowerText);
Myth.CGC.selectUpperTextString = String(Myth.Parameters.selectUpperTextString);
Myth.CGC.selectLowerTextString = String(Myth.Parameters.selectLowerTextString);

Myth.CGC.AP.texts = {
	selectionTopZone: Myth.Parameters.selectUpperTextString,
	selectionTopNoZone: Myth.Parameters.topTextNoZone,
	selectionBottom: Myth.Parameters.selectLowerTextString,
}

Myth.CGC.AP.texts.zoneNames = Myth.Util.tryParse(Myth.Parameters.zoneNames, { "deck": "Deck", "discard": "Discard", "hand": "Hand" }, "MYTH_CGC_ActionPack1");

Myth.CGC.AP.Window_CardSelection_initialize = Window_CardSelection.prototype.initialize;
Window_CardSelection.prototype.initialize = function (cards, direction)
{
	this._cardsSelected = [];
	this._cardsUnselected = [];
	this._indexesSelected = [];
	this.resetSelectTexts();
	Myth.CGC.AP.Window_CardSelection_initialize.call(this, cards, direction);
}

Window_CardSelection.prototype.readySelection = function (index)
{
	var doesContain = false;
	//var selectedCards = this._cardsSelected.getCardSprites();
	var selectedCards = this._indexesSelected;
	for (var i = selectedCards.length - 1; i >= 0; i--)
	{
		if (selectedCards[i] == index)
		{
			doesContain = true;
			this._cardSprites[index]._enabledSprite.hide();
			selectedCards.splice(i, 1);
			this.contents.clear();
			this.drawAllItems();
			break;
			//this._cardSprites.getCardSprites()[index]._readyToDiscard = false;
		}
	}

	if (!doesContain && selectedCards.length < BattleManager._cardsToSelect)
	{
		selectedCards.push(index);
		this._cardSprites[index]._enabledSprite.show();
		this.contents.clear();
		this.drawAllItems();
		//this._cardSprites.getCardSprites()[index]._readyToDiscard = true;
		this.selectSelectedCards();
	}


};

Myth.CGC.AP.Window_CardSelection_createCardSprites = Window_CardSelection.prototype.createCardSprites;
Window_CardSelection.prototype.createCardSprites = function ()
{
	Myth.CGC.AP.Window_CardSelection_createCardSprites.call(this);
	for (var i = 0; i < this._cardSprites.length; i++)
	{
		var cardSprite = this._cardSprites[i];
		if (Myth.CGC.images.cardSelectedHighlight)
		{
			var bitmap = ImageManager.loadBitmap("img/CGC/highlights/", Myth.CGC.images.cardSelectedHighlight)
			cardSprite._enabledSprite.bitmap = bitmap;
			//cardSprite._enabledSprite.setFrame(0, 0, bitmap.width, bitmap.height);
		}
	}
}

Window_CardSelection.prototype.selectSelectedCards = function ()
{
	//var selectedCards = this._cardsSelected.getCardSprites();
	var selectedCards = this._indexesSelected;
	if (selectedCards.length == BattleManager._cardsToSelect)
	{
		for (var i = 0; i < selectedCards.length; i++)
		{
			this._cardsSelected.push(this._cards[selectedCards[i]]);
		}
		BattleManager._cardsToSelect = 0;
		this._indexesSelected = [];
		this._cardsUnselected = this._cards.filter(card => !this._cardsSelected.includes(card));
	}
};

Window_CardSelection.prototype.selectAll = function ()
{
	for (var i = 0; i < this._cards.length; i++)
	{
		this._cardsSelected.push(this._cards[i]);
	}
	BattleManager._cardsToSelect = 0;
	this._indexsSelected = [];
	this._cardsUnselected = [];
};

Window_CardSelection.prototype.selectedCards = function ()
{
	return this._cardsSelected;
}

Window_CardSelection.prototype.unselectedCards = function ()
{
	return this._cardsUnselected;
}
Window_CardSelection.prototype.clearSelectedCards = function ()
{
	this._indexesSelected = [];
	this._cardsSelected = [];
	this._cardsUnselected = [];
}

Window_CardSelection.prototype.drawAllItems = function ()
{
	if (BattleManager._cardsToSelect == 0 || BattleManager._cardsToSelect == undefined) return;
	var texts = Myth.CGC.AP.texts;

	if (Myth.CGC.showSelectionUpperText) this.drawUpperText();
	if (Myth.CGC.showSelectionLowerText) this.drawLowerText();
};

Window_CardSelection.prototype.drawUpperText = function ()
{
	var texts = Myth.CGC.AP.texts;
	var topText = Myth.CGC.selectUpperTextString;

	if (!topText)
	{
		topText = "Choose " + ((BattleManager._isArbitrary) ? "any number of" : BattleManager._cardsToSelect) + " cards from ";

		if (BattleManager._selectedZone == null) {
			topText += "the selection.";
		}
		else {
			topText += BattleManager._currentCardBattler.name() + "'s ";
			switch (BattleManager._selectedZone) {
				case "deck": topText += "Deck."; break;
				case "hand": topText += "Hand."; break;
				case "discard": topText += "Discard."; break;
				default: topText += BattleManager._selectedZone + "."; break;
			}
		}
	}
	else
	{
		var zone = "the selection";
		if (BattleManager._selectedZone)
		{
			zone = texts.zoneNames[BattleManager._selectedZone];
			if (zone == undefined)
				zone = BattleManager._selectedZone;
		}
		var requirement = BattleManager._isArbitrary ? "any" : BattleManager._cardsToSelect.toString();

		topText = topText.replace(/%actor/gi, BattleManager._currentCardBattler.name());
		

		topText = topText.replace(/%selectNum/gi, this._indexesSelected.length);
		topText = topText.replace(/%unselectNum/gi, String(BattleManager._cardsToSelect - this._indexesSelected.length));
		topText = topText.replace(/%selectReq/gi, requirement);
		topText = topText.replace(/%zone/gi, zone);
	}

	this.drawText(topText, 0, 0, this.contents.width, 'center');
}

Window_CardSelection.prototype.drawLowerText = function ()
{
	var texts = Myth.CGC.AP.texts;
	var bottomText = Myth.CGC.selectLowerTextString;

	if (!bottomText)
	{
		bottomText = this._indexesSelected.length + " / " + BattleManager._cardsToSelect + " selected.";
		if (BattleManager._isArbitrary) bottomText = "Chosen cards will be shuffled back into the Deck.";
	}
	else
	{
		var zone = "the selection";
		if (BattleManager._selectedZone)
		{
			zone = texts.zoneNames[BattleManager._selectedZone];
			if (zone == undefined)
				zone = BattleManager._selectedZone;
		}
		var requirement = BattleManager._isArbitrary ? "any" : BattleManager._cardsToSelect.toString();

		bottomText = bottomText.replace(/%actor/gi, BattleManager._currentCardBattler.name());
		bottomText = bottomText.replace(/%selectNum/gi, this._indexesSelected.length);
		bottomText = bottomText.replace(/%unselectNum/gi, String(BattleManager._cardsToSelect - this._indexesSelected.length));
		bottomText = bottomText.replace(/%selectReq/gi, requirement);
		bottomText = bottomText.replace(/%zone/gi, zone);
	}

	this.drawText(bottomText, 0, this.contents.height - this.lineHeight(), this.contents.width, 'center');
}

Window_CardSelection.prototype.resetSelectTexts = function ()
{
	var texts = Myth.CGC.AP.texts;
	Myth.CGC.selectUpperTextString = texts.selectionTopZone;
	Myth.CGC.selectLowerTextString = texts.selectionBottom;
}

Window_CardSelection.prototype.giveCardToRemovedSprites = function (card)
{
	var cardSprite = this._cardSprites.find((sprite) => sprite._card == card);
	var index = this._cardSprites.indexOf(cardSprite);

	var trueX = cardSprite.x + this.x;
	var trueY = cardSprite.y + this.y;
	cardSprite._enabledSprite.hide();
	cardSprite.scale = this.cardScale();

	var skillWindow = SceneManager._scene._skillWindow;
	skillWindow._removedSprites.getCardSprites().push(cardSprite);
	this._cardSprites.splice(index, 1);
	cardSprite.x = trueX;
	cardSprite.y = trueY;

	this.removeChild(cardSprite);
	SceneManager._scene.addCardSprite(cardSprite);
}

Myth.CGC.AP.Scene_Battle_onCardSelectionOk = Scene_Battle.prototype.onCardSelectionOk;
Scene_Battle.prototype.onCardSelectionOk = function ()
{
	if (BattleManager._phase != 'cardSelection')
		Myth.CGC.AP.Scene_Battle_onCardSelectionOk.call(this);

	var index = this._cardSelectionWindow.index();
	this._cardSelectionWindow.readySelection(index);
}

Myth.CGC.AP.Scene_Battle_onCardSelectionCancel = Scene_Battle.prototype.onCardSelectionCancel;
Scene_Battle.prototype.onCardSelectionCancel = function ()
{
	if (BattleManager._phase != 'cardSelection')
		Myth.CGC.AP.Scene_Battle_onCardSelectionCancel.call(this);

	if (BattleManager._isArbitrary)
	{
		BattleManager._cardsToSelect = this._cardSelectionWindow._indexesSelected.length;
		this._cardSelectionWindow.selectSelectedCards();
		BattleManager._isArbitrary = false;
	}
};

Myth.CGC.AP.Scene_Battle_updateBattleProcess = Scene_Battle.prototype.updateBattleProcess;
Scene_Battle.prototype.updateBattleProcess = function ()
{
	if (BattleManager._phase == 'cardSelection')
		BattleManager.updateCardSelection();

	Myth.CGC.AP.Scene_Battle_updateBattleProcess.call(this);
}

Myth.CGC.AP.BattleManager_startBattle = BattleManager.startBattle;
BattleManager.startBattle = function ()
{
	Myth.CGC.AP.BattleManager_startBattle.call(this);
	this._selectedZone = null;
}

BattleManager.clearSelectedCards = function ()
{
	this._selectedZone = null;
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	selectionWindow.clearSelectedCards();
}

BattleManager.requireSelection = function (amount, pool)
{
	if ($gameTroop.isAllDead())
		return;
	if (amount == 0 || pool.length == 0) return;

	this._cardsToSelect = amount || 1;
	this._cardsToSelect = Math.min(this._cardsToSelect, pool.length);
	this._previousPhase = this._phase;
	this._phase = 'cardSelection';
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	selectionWindow.setCards(pool);
	selectionWindow.show();
	selectionWindow.activate();
	selectionWindow.select(0);
	var helpWindow = SceneManager._scene._helpWindow;
	helpWindow.show();

};

BattleManager.requireArbitrarySelection = function (actor, zoneName)
{
	if ($gameTroop.isAllDead())
		return;

	var zone = actor.getZoneByName(zoneName.toLowerCase());
	if (zone == null) return console.error();
	var pool = zone._data;

	this._isArbitrary = true;
	this._cardsToSelect = pool.length;
	this._selectedZone = zoneName;

	this._previousPhase = this._phase;
	this._phase = 'cardSelection';
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	selectionWindow.setCards(pool);
	selectionWindow.show();
	selectionWindow.activate();
	selectionWindow.select(0);
	var helpWindow = SceneManager._scene._helpWindow;
	helpWindow.show();

}

BattleManager.requireSelectionByZone = function (actor, amount, zoneName, poolSize, place, type)
{
	var pool = [];
	var zone = actor.getZoneByName(zoneName.toLowerCase());
	if (zone == null) return console.error();
	this._selectedZone = zoneName.toLowerCase();
	if (poolSize == undefined)
	{
		pool = zone._data;
		if (type != undefined)
		{
			pool = pool.filter(function (card)
			{
				return DataManager.isSkillOfType(card.id(), type);
			});
		}
	}
	else
	{
		place = place.toLowerCase();
		var tempCards = new Game_Cards("temp");
		tempCards.copy(zone);
		if (place == "random")
			tempCards.shuffle();
		else if (place == "bottom")
			tempCards._data = tempCards._data.reverse();
		if (type != undefined)
		{
			tempCards._data = tempCards._data.filter(function (card)
			{
				return DataManager.isSkillOfType(card.id(), type);
			});
			tempCards.length = tempCards._data.length;
		}
		poolSize = Math.min(poolSize, tempCards.length);
		for (var i = 0; i < poolSize; i++)
		{
			pool.push(tempCards.card(i));
		}
	}

	this.requireSelection(amount, pool);
};

BattleManager.requireSelectionByArray = function (amount, skillArray)
{
	var pool = [];
	for (var i = 0; i < skillArray.length; i++)
	{
		var card = new Game_Card(skillArray[i]);
		pool.push(card);
	}

	this.requireSelection(amount, pool);
}

BattleManager.selectAllByZone = function (actor, zoneName, type)
{
	var zone = actor.getZoneByName(zoneName.toLowerCase());
	if (zone == null) return console.error();
	this._selectedZone = zoneName.toLowerCase();
	var pool = zone._data;
	if (type != undefined)
	{
		pool = pool.filter(function (card)
		{
			return DataManager.isSkillOfType(card.id(), type);
		});
	}
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	this._previousPhase = this._phase;
	this._phase = 'cardSelection';
	selectionWindow.setCards(pool);
	selectionWindow.selectAll();
};

BattleManager.updateCardSelection = function ()
{
	if (this._cardsToSelect <= 0)
	{
		this._phase = this._previousPhase;
		var selectionWindow = SceneManager._scene._cardSelectionWindow;
		selectionWindow.hide();
	}
	else
	{
		var selectionWindow = SceneManager._scene._cardSelectionWindow;
		selectionWindow.activate();
	}
}


Myth.CGC.AP.Game_Actor_performCardAction = Game_Battler.prototype.performCardAction;
Game_Battler.prototype.performCardAction = function (action)
{
	var originalAction = action;
	action = this.formatCardAction(action);
	if (action.match(/(?:select )(\d+)(?: from skills )((\d+? ?,? ?)+)/i))
	{
		var amount = Number(RegExp.$1);
		var array = RegExp.$2.split(/[\s,]+/);
		for (var i = 0; i < array.length; i++)
		{
			var num = Number(array[i].trim());
			if (num != NaN)
				array[i] = num;
		}
		BattleManager.requireSelectionByArray(amount, array);
	}
	else if (action.match(/(?:select any from )(\w+)$/i))
	{
		var zone = RegExp.$1;
		BattleManager.requireArbitrarySelection(this, zone);
	}
	else if (action.match(/select upper text:? (.+)/i))
	{
		Myth.CGC.selectUpperTextString = String(RegExp.$1);
	}
	else if (action.match(/select lower text:? (.+)/i))
	{
		Myth.CGC.selectLowerTextString = String(RegExp.$1);
	}
	else if (action.match(/(?:select )(\d+)(?: of )(\d+)(?: from )(\w+) (top|bottom|random)/i))
	{
		var amount = Number(RegExp.$1);
		var poolSize = Number(RegExp.$2)
		var zone = RegExp.$3;
		var place = RegExp.$4;

		BattleManager.requireSelectionByZone(this, amount, zone, poolSize, place);
	}
	else if (action.match(/(?:select )(\d+)(?: from )(\w+)$/i))
	{
		var amount = Number(RegExp.$1);
		var zone = RegExp.$2;

		BattleManager.requireSelectionByZone(this, amount, zone);
	}
	else if (action.match(/(?:select )(\d+)(?: type )(\w+)(?: from )(\w+)$/i))
	{
		var amount = Number(RegExp.$1);
		var type = RegExp.$2;
		var zone = RegExp.$3;

		BattleManager.requireSelectionByZone(this, amount, zone, undefined, undefined, type);
	}
	else if (action.match(/(?:select )(\d+)(?: of )(\d+)(?: type )(\w+)(?: from )(\w+) (top|bottom|random)/i))
	{
		var amount = Number(RegExp.$1);
		var poolSize = Number(RegExp.$2);
		var type = RegExp.$3;
		var zone = RegExp.$4;
		var place = RegExp.$5;

		BattleManager.requireSelectionByZone(this, amount, zone, poolSize, place, type);
	}
	else if (action.match(/(?:select all from )(\w+)/i))
	{

		var zone = RegExp.$1;
		BattleManager.selectAllByZone(this, zone);
	}
	else if (action.match(/select all type (\w+) from (\w+)/i))
	{
		var type = RegExp.$1;
		var zone = RegExp.$2;
		BattleManager.selectAllByZone(this, zone, type);
	}
	else if (action.match(/(?:move (selected|unselected|remaining) to )(\w+)/i))
	{
		var selectedType = RegExp.$1.toLowerCase() == "selected" ? true : false;
		var newZone = RegExp.$2;
		this.moveSelectedCards(selectedType, newZone);
	}
	else if (action.match(/remove (selected|unselected|remaining)/i))
	{
		var selectedType = RegExp.$1.toLowerCase() == "selected" ? true : false;
		this.removeSelectedCards(selectedType);
	}
	else if (action.match(/(?:add (selected|unselected|remaining) to )(\w+)/i))
	{
		var selectedType = RegExp.$1.toLowerCase() == "selected" ? true : false;
		var zone = RegExp.$2;
		var win = SceneManager._scene._cardSelectionWindow;
		var selectedCards = selectedType ? win.selectedCards() : win.unselectedCards();
		this.addCardsToZone(selectedCards, zone);
	}
	else if (action.match(/clear (?:selected|selection)/i))
	{
		BattleManager.clearSelectedCards();
		var win = SceneManager._scene._cardSelectionWindow;
		if(win) win.resetSelectTexts();
	}
	else
		Myth.CGC.AP.Game_Actor_performCardAction.call(this, originalAction);
};


Game_Battler.prototype.addCardsToZone = function (skillArray, zone)
{
	for (var i = 0; i < skillArray.length; i++)
	{
		var skillId = skillArray[i].id();
		this.addCardToZone(skillId, zone);
	}
}

Game_Battler.prototype.moveSelectedCards = function (isSelected, newZone)
{
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	var cards = isSelected ? selectionWindow.selectedCards() : selectionWindow.unselectedCards();
	var oldZone = BattleManager._selectedZone;
	if (oldZone == null) return;

	var startingZone = this.getZoneByName(oldZone);
	var endingZone = this.getZoneByName(newZone);
	if (startingZone.isSpecial)
	{
		return this.moveSelectedCardsSpecial(newZone);
	}

	for (var i = 0; i < cards.length; i++)
	{
		var card = cards[i];
		var index = startingZone.indexOfObject(card);
		this.moveCard(index, startingZone, endingZone);
	}
}

Game_Battler.prototype.removeSelectedCards = function (isSelected)
{
	var selectionWindow = SceneManager._scene._cardSelectionWindow;
	var skillWindow = SceneManager._scene._skillWindow;
	if (!selectionWindow) return;

	var cards = isSelected ? selectionWindow.selectedCards() : selectionWindow.unselectedCards();
	var oldZone = BattleManager._selectedZone;
	if (oldZone == null) return;

	var zone = this.getZoneByName(oldZone);
	if (zone.isSpecial)
	{
		return this.removeSelectedCardsSpecial();
	}

	for (var i = cards.length - 1; i >= 0; i--)
	{
		var card = cards[i];
		var index = zone.indexOfObject(card);
		if (index == -1) continue;

		if (oldZone.toLowerCase() == "hand")
		{

			skillWindow.removeCard(index + skillWindow._itemsBeforeCards);
			//var cardSprite = this._cardSprites.getCardSprites()[index];
			//this._cardSprites.getCardSprites().splice(index, 1);
			//this._removedSprites.getCardSprites().push(cardSprite);
		}
		else
		{
			skillWindow.removeCardSprite(card, oldZone);
			selectionWindow.giveCardToRemovedSprites(card);
			zone.splice(index, 1);
			this.removedCards++;
		}


	}

	this.updateCardVariables();
}

if (Game_Battler.prototype.removeSelectedCardsSpecial == undefined)
{
	Game_Battler.prototype.removeSelectedCardsSpecial = function ()
	{
		//to be overwritten in functions that can be placed above this
	}
}

if (Game_Battler.prototype.moveSelectedCardsSpecial == undefined)
{
	Game_Battler.prototype.moveSelectedCardsSpecial = function (newZone)
	{
		//to be overwritten in functions that can be placed above this
	}
}
